/* ========================================================================== */
/*! \file
 * \brief Test of \c posix_inet_pton() implementation
 *
 * Copyright (c) 2017-2020 by the developers. See the LICENSE file for details.
 */


/* ========================================================================== */
/* Include headers */

#include "posix.h"  /* Include this first because of feature test macros */

#include <stdio.h>

#include "config.h"

#include "test.h"
#include "test_inet_pton.h"


/* ========================================================================== */
/*! \addtogroup TEST */
/*! @{ */


/* ========================================================================== */
/*! \brief Test \c posix_inet_pton() implementation
 *
 * The following cases are tested:
 * - Valid IPv4 address with maximum length
 * - Valid IPv4 address with minimum length
 * - Invalid IPv4 address (must be rejected)
 *
 * \return
 * - \c EXIT_SUCCESS on success
 * - \c EXIT_FAILURE on error
 */

int  test_inet_pton(void)
{
   int  res = POSIX_EXIT_SUCCESS;
   const char*  s;
   CFG_UINT32_TYPE  expected1 = (CFG_UINT32_TYPE) 3232261220UL;
   CFG_UINT32_TYPE  expected2 = (CFG_UINT32_TYPE) 16909060UL;
   posix_in_addr_t  valh, valn;
   int  rv;

   /*
    * This test the following things:
    * - Valid IPv4 address with maximum length
    */
   s = "192.168.100.100";
   valn = 0;
   rv = posix_inet_pton(POSIX_AF_INET, s, &valn);
   if(!rv)
   {
      print_error("Conversion failed");
      res = POSIX_EXIT_FAILURE;

      /* For debugging */
      fprintf(stderr, TEST_TAB "Input: \"%s\"\n", s);
   }
   if(1 == rv)
   {
      valh = posix_ntohl(valn);
      if(expected1 != valh)
      {
         print_error("Result is not correct");
         res = POSIX_EXIT_FAILURE;

         /* For debugging */
         fprintf(stderr, TEST_TAB "Input: \"%s\"\n", s);
         fprintf(stderr, TEST_TAB "Result is: %lu\n",
                 (unsigned long int) valh);
         fprintf(stderr, TEST_TAB "Should be: %lu\n",
                 (unsigned long int) expected1);
      }
   }

   if(!res)
   {
      /*
       * This test the following things:
       * - Valid IPv4 address with minimum length
       */
      s = "1.2.3.4";
      valn = 0;
      rv = posix_inet_pton(POSIX_AF_INET, s, &valn);
      if(!rv)
      {
         print_error("Conversion failed");
         res = POSIX_EXIT_FAILURE;

         /* For debugging */
         fprintf(stderr, TEST_TAB "Input: \"%s\"\n", s);
      }
      if(1 == rv)
      {
         valh = posix_ntohl(valn);
         if(expected2 != valh)
         {
            print_error("Result is not correct");
           res = POSIX_EXIT_FAILURE;

            /* For debugging */
            fprintf(stderr, TEST_TAB "Input: \"%s\"\n", s);
            fprintf(stderr, TEST_TAB "Result is: %lu\n",
                    (unsigned long int) valh);
            fprintf(stderr, TEST_TAB "Should be: %lu\n",
                    (unsigned long int) expected2);
         }
      }
   }

   if(!res)
   {
      /*
       * This test the following things:
       * - Invalid IPv4 address
       */
      s = "x192.168.100.100";
      valn = 0;
      rv = posix_inet_pton(POSIX_AF_INET, s, &valn);
      if(rv)
      {
         print_error("Conversion succeeded for invalid input");
         res = POSIX_EXIT_FAILURE;

         /* For debugging */
         fprintf(stderr, TEST_TAB "Input: \"%s\"\n", s);
      }
   }

   return(res);
}


/*! @} */

/* EOF */
