#ifndef NNTP_H
#define NNTP_H  1

/* ========================================================================== */
/* Include files */

#include <sys/types.h>
#include <limits.h>


/*! \addtogroup NNTP */
/*! @{ */


/* ========================================================================== */
/* Data types */

/*! \brief Article number
 *
 * RFC 3977 requires at least 31 bit storage for article numbers.
 * The constant \ref NNTP_ANUM_T_MAX must be set to the maximum value that
 * is supported by this implementation.
 * According to RFC 3977, article number zero is reserved.
 *
 * \attention
 * This must always be an unsigned integer type.
 */
typedef unsigned long int  nntp_anum_t;

/*! \brief NNTP group descriptor */
struct nntp_groupdesc
{
   char*  name;               /*!< Group name */
   nntp_anum_t  eac;          /*!< Estimated article count */
   nntp_anum_t  lwm;          /*!< Low water mark */
   nntp_anum_t  hwm;          /*!< High water mark */
   unsigned int  flags;       /*!< Flags (use \c NNTP_GROUP_xxx constants) */
};


/* ========================================================================== */
/* Constants */

/*! \brief Maximum value this implementation supports for \ref nntp_anum_t */
#define NNTP_ANUM_T_MAX  ULONG_MAX

/*! \name Connection closing flags */
/*! @{ */
/*! Don't send quit command before closing connection to server */
#define NNTP_CLOSE_NOQUIT  1U
/*! @} */

/*! \name Newsgroup flags */
/*! @{ */
/*! Group name contains only printable ASCII characters */
#define NNTP_GROUP_FLAG_ASCII  0x01U
/*! Posting to group is allowed */
#define NNTP_GROUP_FLAG_PA     0x02U
/*! @} */


/*! @} */


/* ========================================================================== */
/* Function prototypes */

int  nntp_open(int*, const char*, const char*, const char*, int, int, ...);
void  nntp_close(int*, unsigned int);
int  nntp_get_capa_list_motd(int);
int  nntp_get_motd(int, char**, size_t*);
int  nntp_get_distrib_pats(int, const char**, size_t*);
struct nntp_groupdesc*  nntp_group_descriptor_constructor(const char*);
int  nntp_get_grouplist(int, size_t*, struct nntp_groupdesc**);
int  nntp_set_group(int, const char*, struct nntp_groupdesc**);
int  nntp_get_capa_over(int);
int  nntp_get_overview(int, nntp_anum_t, nntp_anum_t, char**, size_t*);
int  nntp_get_article_by_mid(int, const char*, char**, size_t*);
int  nntp_get_article(int, const nntp_anum_t*, char**, size_t*);
int  nntp_get_article_header(int, const nntp_anum_t*, char**, size_t*);
int  nntp_get_article_body(int, const nntp_anum_t*, char**, size_t*);
int  nntp_post_article(int, const char*);


#endif  /* NNTP_H */

/* EOF */
