#ifndef CONF_H
#define CONF_H  1


/*! \addtogroup CONF */
/*! @{ */


/* ========================================================================== */
/* Constants */

/*! \brief Number of entries in global configuration array */
#define CONF_NUM  57


/* ========================================================================== */
/* Data types */

/*! \brief IDs for use as index in configuration array
 *
 * \attention
 * Don't forget to modify the number of entries \ref CONF_NUM after adding or
 * removing a configuration entry!
 *
 * \note
 * A C90 compiler is required to assign the values starting from zero without
 * holes by default. We force the default behaviour explicitly only to make
 * it more obvious for humans what values are used.
 */
enum conf_entry
{
   CONF_POS_X = 0,                 /*!< Main window X position */
   CONF_POS_Y = 1,                 /*!< Main window Y position */
   CONF_SIZE_X = 2,                /*!< Main window X size */
   CONF_SIZE_Y = 3,                /*!< Main window Y size */
   CONF_TILE_X = 4,                /*!< Main window X tiling */
   CONF_TILE_Y = 5,                /*!< Main window Y tiling */
   CONF_SERVER = 6,                /*!< NNTP server hostname or IP address */
   CONF_SERVICE = 7,               /*!< NNTP service name or TCP port */
   CONF_ENC = 8,                   /*!< NNTP connection encryption algorithm */
   CONF_AUTH = 9,                  /*!< NNTP server authentication algorithm */
   CONF_IMMEDAUTH = 10,            /*!< Flag for immediate authentication */
   CONF_USER = 11,                 /*!< NNTP server authentication account */
   CONF_PASS = 12,                 /*!< NNTP server authentication password */
   CONF_CAC = 13,                  /*!< Clamp article count to this value */
   CONF_FROM = 14,                 /*!< "From" name and E-Mail address */
   CONF_REPLYTO = 15,              /*!< "Reply-To" name and E-Mail address */
   CONF_FQDN = 16,                 /*!< Fully qualified domain name (FQDN) */
   CONF_NEWSRC = 17,               /*!< Pathname of shared newsrc file */
   CONF_SCORERC = 18,              /*!< Pathname of shared scorerc file */
   CONF_INTRO = 19,                /*!< Introduction line format */
   CONF_ORGANIZATION = 20,         /*!< Organization name poster is part of */
   CONF_TVIEW = 21,                /*!< Flag for threaded view */
   CONF_ONLYUR = 22,               /*!< Flag for only unread articles */
   CONF_CANCELKEY = 23,            /*!< Secret for Cancel-Key */
   CONF_EDITOR = 24,               /*!< Pathname of external editor */
   CONF_PPROC = 25,                /*!< Pathname of external postprocessor */
   CONF_INEWS = 26,                /*!< Pathname of external inews */
   CONF_QUOTESTYLE = 27,           /*!< Quoting style (for citation) */
   CONF_QUOTEUNIFY = 28,           /*!< Unification of quoting style */
   CONF_TESTGRP_ERE = 29,          /*!< ERE pattern to match test groups */
   CONF_TESTGRP_KWORDS = 30,       /*!< Keywords to set for test groups */
   CONF_TLS_OWNCERTS = 31,         /*!< Use local TLS root certificates */
   CONF_CRL_CHECK = 32,            /*!< Use TLS certificate CRL check */
   CONF_CRL_UPD_IV = 33,           /*!< TLS certificate CRL update interval */
   CONF_CRL_UPD_TS = 34,           /*!< TLS certificate CRL update timestamp */
   CONF_CRL_UPD_C = 35,            /*!< TLS certificate CRL update choice */
   CONF_UTVIEW_AN = 36,            /*!< Sort unthreaded view by article num. */
   CONF_INV_ORDER = 37,            /*!< Sort articles with inverted order */
   CONF_NO_OVER = 38,              /*!< NNTP driver doesn't use overview */
   CONF_DIST_SUGG = 39,            /*!< Use distribution suggestions */
   CONF_COMPRESSION = 40,          /*!< NNTP driver negotiates compression */
   CONF_SIGFILE = 41,              /*!< Signature file */
   CONF_TS_LTIME = 42,             /*!< Use local time for timestamps */
   CONF_TS_COMMENT = 43,           /*!< Generate timestamps with TZ comment */
   CONF_FLOWED_CRLF = 44,          /*!< Generate empty line after paragraphs */
   CONF_FORCE_UNICODE = 45,        /*!< Force Unicode for outgoing articles */
   CONF_SEARCH_CASE_IS = 46,       /*!< Search case-insensitive */
   CONF_ENABLE_UAGENT = 47,        /*!< Enable User-Agent header field */
   CONF_INITIAL_GREETING = 48,     /*!< Initial greeting phrase */
   CONF_REFRESH_INTERVAL = 49,     /*!< Group list refresh interval */
   CONF_UNREAD_IN_NEXT_GROUP = 50, /*!< Skip to next group for unread article */
   CONF_COLOR_SIGNATURE = 51,      /*!< Signature color */
   CONF_COLOR_EXTERNAL = 52,       /*!< External citation color */
   CONF_COLOR_LEVEL1 = 53,         /*!< Citation color level 1 */
   CONF_COLOR_LEVEL2 = 54,         /*!< Citation color level 2 */
   CONF_COLOR_LEVEL3 = 55,         /*!< Citation color level 3 */
   CONF_COLOR_LEVEL4 = 56          /*!< Citation color level 4 */
   /* !!! Set 'CONF_NUM' above to the first unused value !!! */
};

/*! \brief Data types of configuration array entries */
enum conf_entry_type
{
   CONF_TYPE_INT,                  /*!< Entry is of integer type */
   CONF_TYPE_STRING                /*!< Entry is of string type */
};

/*! \brief Configuration array entry value */
union conf_entry_val
{
   int  i;                         /*!< Integer entry value */
   char*  s;                       /*!< String entry value */
};

/*! \brief Configuration array entry */
struct conf
{
   const char*  label;             /*!< Name before the colon */
   enum conf_entry_type  type;     /*!< Type of entry */
   union conf_entry_val  val;      /*!< Value of the entry behind the colon */
   int  found;                     /*!< Flag indicating entry in config file */
};


/*! @} */


/* ========================================================================== */
/* Variables */

extern struct conf  config[CONF_NUM];
extern int  conf_ephemeral_passwd;


/* ========================================================================== */
/* Function prototypes */

int  conf_load(struct conf*);
int  conf_store(struct conf*);
void  conf_delete(struct conf*);
int  conf_string_replace(struct conf*, const char*);
int  conf_integer_check(enum conf_entry, int, int);


#endif  /* CONF_H */

/* EOF */
